import { NextRequest, NextResponse } from "next/server";
import { editAnnonce } from "@/lib/services/annonceService";
import { PublicationStatus, RealEstateStatus, Type } from "@prisma/client";

const isOneOf = <T extends string>(v: unknown, allowed: readonly T[]): v is T =>
    typeof v === "string" && (allowed as readonly string[]).includes(v);

const toDateOrNull = (v: unknown): Date | null => {
    if (v instanceof Date) return isNaN(v.getTime()) ? null : v;
    if (typeof v === "string" && v.trim()) {
        const d = new Date(v);
        return isNaN(d.getTime()) ? null : d;
    }
    return null;
};

export async function PUT(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) {
            return NextResponse.json({ message: "ID invalide." }, { status: 400 });
        }

        const contentType = req.headers.get("content-type") || "";
        if (!contentType.includes("application/json")) {
            return NextResponse.json({ message: "Content-Type attendu: application/json." }, { status: 415 });
        }

        const body = await req.json();

        const data = {
            title: typeof body.title === "string" ? body.title : undefined,
            address: typeof body.address === "string" ? body.address : undefined,
            country: typeof body.country === "string" ? body.country : undefined,
            city: typeof body.city === "string" ? body.city : undefined,
            postalCode: typeof body.postalCode === "string" ? body.postalCode : undefined,
            description: typeof body.description === "string" ? body.description : undefined,
            surface: typeof body.surface === "number" ? body.surface : undefined,
            price: typeof body.price === "number" ? body.price : undefined,

            publicationStatus: isOneOf<PublicationStatus>(body.publicationStatus, ["UNPUBLISHED", "PUBLISHED"])
                ? body.publicationStatus
                : undefined,

            realeSteateStatus: isOneOf<RealEstateStatus>(body.realeSteateStatus, [
                "FORSALE",
                "SOLD",
                "AVAILABLE",
                "RENTED",
            ])
                ? body.realeSteateStatus
                : undefined,

            type: isOneOf<Type>(body.type, ["RENT", "SALE"]) ? body.type : undefined,

            avaibleFrom: (() => {
                const d = toDateOrNull(body.avaibleFrom);
                return d ?? undefined;
            })(),
        } as const;

        for (const [k, v] of Object.entries(data)) {
            if (v === undefined) {
                return NextResponse.json({ message: `Champ invalide ou manquant: ${k}.` }, { status: 400 });
            }
        }

        const updated = await editAnnonce(id, data);
        return NextResponse.json(updated, { status: 200 });
    } catch (e: any) {
        console.error("Erreur API (PUT /api/annonces/[id]) :", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne du serveur." }, { status: 500 });
    }
}

export async function PATCH(req: NextRequest, { params }: { params: { id: string } }) {
    try {
        const id = Number(params.id);
        if (!Number.isFinite(id)) {
            return NextResponse.json({ message: "ID invalide." }, { status: 400 });
        }

        const contentType = req.headers.get("content-type") || "";
        if (!contentType.includes("multipart/form-data")) {
            return NextResponse.json({ message: "Content-Type attendu: multipart/form-data." }, { status: 415 });
        }

        const form = await req.formData();
        const file = form.get("imagePrincipale");

        if (!file || !(file instanceof File)) {
            return NextResponse.json(
                { message: "Aucun fichier fourni sous le champ 'imagePrincipale'." },
                { status: 400 }
            );
        }

        const ALLOWED = new Set(["image/jpeg", "image/png", "image/webp"]);
        if (!ALLOWED.has(file.type)) {
            return NextResponse.json({ message: "Type d'image non autorisé." }, { status: 415 });
        }

        const buffer = Buffer.from(await file.arrayBuffer());
        const base64Image = buffer.toString("base64");
        const updated = await editAnnonce(id, { mainImg: base64Image });
        return NextResponse.json(updated, { status: 200 });
    } catch (e: any) {
        console.error("Erreur API (PATCH /api/annonces/[id]) :", e);
        return NextResponse.json({ message: e?.message ?? "Erreur interne du serveur." }, { status: 500 });
    }
}
